<?php
// dashboard.php — Role-aware AIT LMIS dashboard + Training Status widget (DRY/KISS)
ini_set('display_errors',1); error_reporting(E_ALL);

require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';

require_login();
$u = current_user();

/* ---------- helpers ---------- */
function qv($sql, $params = []){ $s = db()->prepare($sql); $s->execute($params); return $s->fetch(); }
function qa($sql, $params = []){ $s = db()->prepare($sql); $s->execute($params); return $s->fetchAll(); }
function qn($sql, $params = []){ try { $r = qv($sql,$params); return (int)array_values($r ?? [0])[0]; } catch(Throwable $e){ return 0; } }
function has_table($table){
  try { db()->query("SELECT 1 FROM `$table` LIMIT 1"); return true; }
  catch(Throwable $e){ return false; }
}
function has_col($table,$col){
  try { $st=db()->prepare("SHOW COLUMNS FROM `$table` LIKE ?"); $st->execute([$col]); return (bool)$st->fetch(); }
  catch(Throwable $e){ return false; }
}

/* ---------- existence flags ---------- */
$hasEnroll   = has_table('enrollments');
$hasResults  = has_table('results');
$hasSubs     = has_table('submissions');
$hasModules  = has_table('modules');

/* ---------- common totals (safe) ---------- */
$totalCourses   = qn("SELECT COUNT(*) FROM courses");
$totalLearners  = qn("SELECT COUNT(*) FROM users WHERE role='learner'");
$totalAssessors = qn("SELECT COUNT(*) FROM users WHERE role='assessor'");
$totalCerts     = qn("SELECT COUNT(*) FROM certificates");

$cards=[]; $sections=[]; $quick=[];

/* ---------- LEARNER view ---------- */
if ($u['role']==='learner') {

  // "My courses"
  $myCourses = [];
  if ($hasEnroll){
    $myCourses = qa("SELECT c.id, c.title
                      FROM enrollments e JOIN courses c ON c.id=e.course_id
                     WHERE e.user_id=? ORDER BY c.title", [$u['id']]);
  } elseif ($hasSubs){
    $myCourses = qa("SELECT DISTINCT c.id, c.title
                       FROM submissions s JOIN courses c ON c.id=s.course_id
                      WHERE s.user_id=? ORDER BY c.title", [$u['id']]);
  }

  // Build simple progress cards using modules/submissions if available (fallback)
  foreach ($myCourses as $mc){
    $mods = ($hasModules) ? qn("SELECT COUNT(*) FROM modules WHERE course_id=?", [$mc['id']]) : 0;
    $subm = ($hasSubs)    ? qn("SELECT COUNT(*) FROM submissions WHERE user_id=? AND course_id=?", [$u['id'],$mc['id']]) : 0;
    $pct  = ($mods>0) ? min(100, round(($subm/$mods)*100)) : (($subm>0)? 50 : 0);

    $sections[] = [
      'title'=>$mc['title'],
      'body'=>"
        <div class='progress'><div class='progress__bar' style='width:{$pct}%'></div></div>
        <div class='small'>Progress: <strong>{$pct}%</strong> ".($mods? "· {$subm}/{$mods} submissions":"")."</div>
        <div class='spacer8'></div>
        <a class='btn btn-small' href='".e(url('course_view.php?id='.$mc['id']))."'>Open course</a>
        <a class='btn btn-small btn-outline' href='".e(url('upload_poe.php?course_id='.$mc['id']))."'>Upload POE</a>
      ",
    ];
  }
  
  $cards = [
    ['title'=>'My Courses','value'=>count($myCourses),'hint'=>'Enrolled/active','link'=>url('courses.php')],
    ['title'=>'Certificates','value'=>qn("SELECT COUNT(*) FROM certificates WHERE user_id=?",[$u['id']]),'hint'=>'Issued to you','link'=>url('my_certs.php')],
    ['title'=>'AI Tutor','value'=>'Ask','hint'=>'Instant help','link'=>url('ai_tutor.php')],
  ];
  $quick = [
    ['label'=>'View Courses','href'=>url('courses.php')],
    ['label'=>'Upload POE','href'=>url('upload_poe.php')],
    ['label'=>'Ask AI Tutor','href'=>url('ai_tutor.php')],
  ];

  if (empty($myCourses)) {
    $sections[] = [
      'title'=>'Get started',
      'body'=>"<p>You have no active courses yet.</p>
               <a class='btn btn-small' href='".e(url('courses.php'))."'>Browse courses</a>"
    ];
  }
}

require_once __DIR__ . '/lib/demo.php';
demo_dashboard_boost($cards, $sections, $quick);


/* ---------- ASSESSOR / ADMIN view ---------- */
if ($u['role']==='assessor' || $u['role']==='admin') {

  // Pending POEs
  if ($hasSubs){
    if (has_col('submissions','status')) {
      $pendingPOE = qn("SELECT COUNT(*) FROM submissions WHERE status IN ('submitted','needs_review')");
    } else {
      $pendingPOE = qn("SELECT COUNT(*) FROM submissions");
    }
  } else $pendingPOE = 0;

  // NYC to verify
  if ($hasResults){
    $toVerifyNYC = has_col('results','verified_at')
      ? qn("SELECT COUNT(*) FROM results WHERE decision='NYC' AND verified_at IS NULL")
      : qn("SELECT COUNT(*) FROM results WHERE decision='NYC'");
  } else $toVerifyNYC = 0;

  // Ready to issue
  if ($hasResults){
    if (has_col('results','verified_at')){
      $toIssueCerts = qn("
        SELECT COUNT(*) FROM results r
         WHERE r.decision='C' AND r.verified_at IS NOT NULL
           AND NOT EXISTS(SELECT 1 FROM certificates c WHERE c.user_id=r.user_id AND c.course_id=r.course_id)
      ");
    } else {
      $toIssueCerts = qn("
        SELECT COUNT(*) FROM results r
         WHERE r.decision='C'
           AND NOT EXISTS(SELECT 1 FROM certificates c WHERE c.user_id=r.user_id AND c.course_id=r.course_id)
      ");
    }
  } else $toIssueCerts = 0;

  $cards = [
    ['title'=>'Pending POEs','value'=>$pendingPOE,'hint'=>$hasSubs?'Need grading':'No submissions table','link'=> $hasSubs? url('results.php?tab=pending') : url('reports.php')],
    ['title'=>'Verify NYC','value'=>$toVerifyNYC,'hint'=>$hasResults?'Check borderline':'No results table','link'=> $hasResults? url('results.php?tab=nyc') : url('reports.php')],
    ['title'=>'Issue Certificates','value'=>$toIssueCerts,'hint'=>$hasResults?'Ready to issue':'No results table','link'=> $hasResults? url('results.php?tab=issue') : url('certificates_list.php')],
  ];

  // Recent submissions
  $recent = [];
  if ($hasSubs){
    $recent = qa("
      SELECT s.id, u.name AS learner, c.title AS course, DATE_FORMAT(s.created_at,'%Y-%m-%d') AS d
        FROM submissions s
        JOIN users u ON u.id=s.user_id
        JOIN courses c ON c.id=s.course_id
       ORDER BY s.created_at DESC
       LIMIT 6
    ");
  }
  $html = "<table class='mini'><tr><th>Learner</th><th>Course</th><th>Date</th><th></th></tr>";
  foreach ($recent as $r){
    $html .= "<tr>
      <td>".e($r['learner'])."</td>
      <td>".e($r['course'])."</td>
      <td>".e($r['d'])."</td>
      <td><a class='btn btn-small' href='".e(url('grade.php?id='.$r['id']))."'>Grade</a></td>
    </tr>";
  }
  $html .= "</table>";
  $sections[] = [
    'title'=>'Recent submissions',
    'body'=> $recent ? $html : '<p class="small">No recent submissions (or table missing).</p>',
  ];

  $quick = [
    ['label'=>'Open Grading Queue','href'=> $hasSubs? url('results.php?tab=pending') : url('reports.php')],
    ['label'=>'AI Draft Feedback','href'=>url('ai_feedback.php')],
    ['label'=>'Certificates','href'=>url('certificates_list.php')],
    ['label'=>'Reports','href'=>url('reports.php')],
  ];
}

/* ---------- MODERATOR / ADMIN ---------- */
if ($u['role']==='moderator' || $u['role']==='admin') {
  $toSample = $hasResults
    ? (has_col('results','verified_at') && has_col('results','moderated_at')
        ? qn("SELECT COUNT(*) FROM results WHERE verified_at IS NOT NULL AND moderated_at IS NULL")
        : qn("SELECT COUNT(*) FROM results"))
    : 0;
  $locked = ($hasResults && has_col('results','moderated_at')) ? qn("SELECT COUNT(*) FROM results WHERE moderated_at IS NOT NULL") : 0;

  $sections[] = [
    'title'=>'Moderation queue',
    'body'=>"<p>To sample/lock: <strong>{$toSample}</strong><br>Already locked: <strong>{$locked}</strong></p>
             <a class='btn btn-small' href='".e(url('moderation.php'))."'>Open moderation</a>",
  ];
}

/* ---------- TETA Read-only ---------- */
if ($u['role']==='teta_readonly') {
  $cards = [
    ['title'=>'Courses','value'=>$totalCourses,'hint'=>'Registered courses','link'=>url('courses.php')],
    ['title'=>'Learners','value'=>$totalLearners,'hint'=>'Active learners','link'=>url('reports.php?view=learners')],
    ['title'=>'Certificates','value'=>$totalCerts,'hint'=>'Issued to date','link'=>url('certificates_list.php')],
  ];
  $sections[] = [
    'title'=>'Compliance snapshots',
    'body'=>"<ul class='bullets'>
      <li>Role-based access (learner/assessor/moderator/TETA)</li>
      <li>QR + hash on certificates (verify online)</li>
      <li>POE uploads + audit trail on grading</li>
      <li>Exports: CSV / Excel / PDF</li>
    </ul>",
  ];
  $quick = [
    ['label'=>'Open Reports','href'=>url('reports.php')],
    ['label'=>'Certificates','href'=>url('certificates_list.php')],
    ['label'=>'Verify a Certificate','href'=>url('verify.php')],
  ];
}

/* ---------- Training Status widget (all roles; learners see only self) ---------- */
if ($hasEnroll) {
  $see_all = ($u['role']==='admin' || $u['role']==='assessor' || $u['role']==='moderator' || $u['role']==='report');

  // WHERE scope
  $where = $see_all ? '1=1' : 'e.user_id = ?';
  $p = $see_all ? [] : [$u['id']];

  // Completed today
  $completed = qa("
    SELECT e.user_id, u.name AS learner, c.title AS course, e.completed_at
      FROM enrollments e
      JOIN users u   ON u.id=e.user_id
      JOIN courses c ON c.id=e.course_id
     WHERE $where AND e.completed_at >= CURDATE()
     ORDER BY e.completed_at DESC
     LIMIT 10
  ", $p);

  // Still in progress
  $progressing = qa("
    SELECT e.user_id, u.name AS learner, c.title AS course, e.progress_pct, e.last_activity
      FROM enrollments e
      JOIN users u   ON u.id=e.user_id
      JOIN courses c ON c.id=e.course_id
     WHERE $where
       AND e.completed_at IS NULL
       AND (e.progress_pct IS NOT NULL AND e.progress_pct BETWEEN 1 AND 99)
     ORDER BY COALESCE(e.last_activity, '1970-01-01') DESC
     LIMIT 10
  ", $p);

  // No activity in 7 days
  $stale = qa("
    SELECT e.user_id, u.name AS learner, c.title AS course, e.progress_pct, e.last_activity
      FROM enrollments e
      JOIN users u   ON u.id=e.user_id
      JOIN courses c ON c.id=e.course_id
     WHERE $where
       AND e.completed_at IS NULL
       AND (e.last_activity IS NULL OR e.last_activity < (NOW() - INTERVAL 7 DAY))
     ORDER BY COALESCE(e.last_activity, '1970-01-01') ASC
     LIMIT 10
  ", $p);

  // Counts
  $cntCompleted = qn("SELECT COUNT(*) FROM enrollments e WHERE $where AND e.completed_at >= CURDATE()", $p);
  $cntProgress  = qn("SELECT COUNT(*) FROM enrollments e WHERE $where AND e.completed_at IS NULL AND (e.progress_pct IS NOT NULL AND e.progress_pct BETWEEN 1 AND 99)", $p);
  $cntStale     = qn("SELECT COUNT(*) FROM enrollments e WHERE $where AND e.completed_at IS NULL AND (e.last_activity IS NULL OR e.last_activity < (NOW() - INTERVAL 7 DAY))", $p);

  // Build widget HTML
  ob_start();
  ?>
  <div class="panel">
    <h2 class="panel__title">Training Status</h2>
    <div class="panel__body">

      <!-- Completed Today -->
      <h3 style="margin:8px 0;">Completed today <span class="muted">(<?= (int)$cntCompleted ?>)</span></h3>
      <?php if (!$completed): ?>
        <div class="small muted">No completions so far today.</div>
      <?php else: ?>
        <table class="mini">
          <tr><th>Learner</th><th>Course</th><th>Completed at</th></tr>
          <?php foreach($completed as $r): ?>
            <tr>
              <td><?= e($r['learner']) ?></td>
              <td><?= e($r['course']) ?></td>
              <td><?= e($r['completed_at']) ?></td>
            </tr>
          <?php endforeach; ?>
        </table>
      <?php endif; ?>

      <div class="spacer8"></div>

      <!-- Still in Progress -->
      <h3 style="margin:8px 0;">Still in progress <span class="muted">(<?= (int)$cntProgress ?>)</span></h3>
      <?php if (!$progressing): ?>
        <div class="small muted">No learners currently in progress.</div>
      <?php else: ?>
        <table class="mini">
          <tr><th>Learner</th><th>Course</th><th>Progress</th><th>Last activity</th></tr>
          <?php foreach($progressing as $r): ?>
            <tr>
              <td><?= e($r['learner']) ?></td>
              <td><?= e($r['course']) ?></td>
              <td><?= (int)$r['progress_pct'] ?>%</td>
              <td><?= e($r['last_activity'] ?: '—') ?></td>
            </tr>
          <?php endforeach; ?>
        </table>
      <?php endif; ?>

      <div class="spacer8"></div>

      <!-- No activity in 7 days -->
      <h3 style="margin:8px 0;">No activity in 7 days <span class="muted">(<?= (int)$cntStale ?>)</span></h3>
      <?php if (!$stale): ?>
        <div class="small muted">Everyone is active 🎉</div>
      <?php else: ?>
        <table class="mini">
          <tr><th>Learner</th><th>Course</th><th>Progress</th><th>Last activity</th></tr>
          <?php foreach($stale as $r): ?>
            <tr>
              <td><?= e($r['learner']) ?></td>
              <td><?= e($r['course']) ?></td>
              <td><?= (int)($r['progress_pct'] ?? 0) ?>%</td>
              <td><?= e($r['last_activity'] ?: '—') ?></td>
            </tr>
          <?php endforeach; ?>
        </table>
      <?php endif; ?>

    </div>
  </div>
  <?php
  $sections[] = ['title'=>'','body'=>ob_get_clean()];
}

/* ---------- ADMIN extras ---------- */
if ($u['role']==='admin') {
  $sections[]=[
    'title'=>'Admin quick links',
    'body'=>"<div class='chips'>
      <a class='chip' href='".e(url('courses.php'))."'>Manage Courses</a>
      <a class='chip' href='".e(url('users.php'))."'>Manage Users</a>
      <a class='chip' href='".e(url('settings_provider.php'))."'>Branding & Settings</a>
      <a class='chip' href='".e(url('reports.php'))."'>Reports</a>
    </div>"
  ];
}

/* ---------- render ---------- */
include __DIR__ . '/templates/header.php';
?>
<h1 style="margin-bottom:6px;">Hello, <?= e($u['name']) ?></h1>
<p class="small" style="margin-top:0;">Your <?= e(strtoupper($u['role'])) ?> dashboard</p>

<div class="dash-cards">
  <?php if (empty($cards)): ?>
    <div class="card">
      <h3>Welcome</h3>
      <p>Use the navigation above to get started.</p>
    </div>
  <?php else: foreach ($cards as $c): ?>
    <article class="card card--stat">
      <div class="stat__value"><?= e($c['value']) ?></div>
      <div class="stat__label"><?= e($c['title']) ?></div>
      <div class="small muted"><?= e($c['hint'] ?? '') ?></div>
      <?php if (!empty($c['link'])): ?>
        <div class="spacer8"></div>
        <a class="btn btn-small" href="<?= e($c['link']) ?>">Open</a>
      <?php endif; ?>
    </article>
  <?php endforeach; endif; ?>
</div>

<?php foreach ($sections as $s): ?>
  <section class="panel">
    <?php if (!empty($s['title'])): ?><h2 class="panel__title"><?= e($s['title']) ?></h2><?php endif; ?>
    <div class="panel__body"><?= $s['body'] ?></div>
  </section>
<?php endforeach; ?>

<?php
require_once __DIR__.'/lib/recommend.php'; // make sure you created this helper file earlier

// show driver score and recommended course
$ls = latest_driver_score($u['id'], 'weekly');
if ($ls) {
  $rec = recommended_course_for_score((int)$ls['score']);
  $recHtml = $rec
    ? "<a class='btn btn-small' href='".e(url('course_view.php?id='.$rec['course_id']))."'>Start: ".e($rec['title'])."</a>"
    : "<span class='small muted'>No recommendation yet.</span>";

  $sections[] = [
    'title' => 'Your Driving Safety Score',
    'body'  => "<p>Latest weekly score: <strong>".(int)$ls['score']."</strong></p>".$recHtml
  ];
}
?>


<?php if (!empty($quick)): ?>
  <section class="panel">
    <h2 class="panel__title">Quick actions</h2>
    <div class="chips">
      <?php foreach ($quick as $q): ?>
        <a class="chip" href="<?= e($q['href']) ?>"><?= e($q['label']) ?></a>
      <?php endforeach; ?>
    </div>
  </section>
<?php endif; ?>

<?php include __DIR__ . '/templates/footer.php'; ?>
