<?php
// certificate_pdf.php — render certificate HTML as a single-page PDF (A4)
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';
require_once __DIR__ . '/lib/settings.php';
require_once __DIR__ . '/lib/pii.php';
require_once __DIR__ . '/vendor/autoload.php';

use Dompdf\Dompdf;
use Dompdf\Options;

/* 1) Read & validate code */
$code = trim($_GET['code'] ?? '');
if ($code === '') {
  http_response_code(400);
  echo "Missing certificate code.";
  exit;
}

/* 2) Load certificate + learner + course (one query) */
$st = db()->prepare("
  SELECT
    c.*,
    u.id               AS u_id,
    u.name             AS u_name,
    u.email            AS u_email,
    u.national_id_enc  AS u_nid_enc,
    co.id              AS co_id,
    co.title           AS co_title,
    co.nqf_level       AS co_nqf,
    co.credits         AS co_credits,
    co.course_number   AS co_number,
    co.expiry_months   AS co_expiry
  FROM certificates c
  JOIN users    u  ON u.id  = c.user_id
  JOIN courses  co ON co.id = c.course_id
  WHERE c.cert_code = ?
  LIMIT 1
");
$st->execute([$code]);
$row = $st->fetch(PDO::FETCH_ASSOC);

if (!$row) {
  http_response_code(404);
  echo "Certificate not found.";
  exit;
}

/* 3) Build arrays the template expects */
$CERT = [
  'id'               => $row['id'],
  'cert_code'        => $row['cert_code'],
  'issued_at'        => $row['issued_at'],
  'assessor_reg_no'  => $row['assessor_reg_no'] ?? null,
  'assessor_name'    => $row['assessor_name']   ?? null,
  'principal_name'   => $row['principal_name']  ?? null,
];

$LEARN = [
  'id'               => $row['u_id'],
  'name'             => $row['u_name'],
  'email'            => $row['u_email'],
  'national_id_enc'  => $row['u_nid_enc'],
];

$COURSE = [
  'id'            => $row['co_id'],
  'title'         => $row['co_title'],
  'nqf_level'     => $row['co_nqf'],
  'credits'       => $row['co_credits'],
  'course_number' => $row['co_number'],
  'expiry_months' => $row['co_expiry'],
];

/* 4) Enforce SA ID (13 digits) AND pass it to the template */
$learner_id = '';
if (!empty($LEARN['national_id_enc'])) {
  $learner_id = pii_decrypt($LEARN['national_id_enc']);
}
if (!preg_match('/^\d{13}$/', $learner_id)) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Certificate blocked</h1>";
  echo "<div class='alert alert-error'>
          Learner South African ID (13 digits) is missing or invalid.<br>
          Please update the learner’s <a href='".e(url('profile.php'))."'>profile</a> and try again.
        </div>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

/* 5) Tell the template we’re rendering for PDF */
$for_pdf = true;

/* 6) Render the SAME HTML as the web view */
ob_start();
include __DIR__ . '/templates/certificate_render.php';  // uses $CERT, $LEARN, $COURSE, $learner_id, $for_pdf
$html = ob_get_clean();

/* 7) Dompdf setup */
$opts = new Options();
$opts->set('isRemoteEnabled', true);
$opts->set('isHtml5ParserEnabled', true);
$opts->set('defaultFont', 'DejaVu Sans');
$opts->set('dpi', 96);
$opts->setChroot(__DIR__);

$dompdf = new Dompdf($opts);
$dompdf->setPaper('A4', 'portrait');
$dompdf->loadHtml($html, 'UTF-8');
$dompdf->render();

/* 8) Stream inline */
$filename = 'certificate_' . preg_replace('~[^A-Za-z0-9_-]+~', '', $code) . '.pdf';
$dompdf->stream($filename, ['Attachment' => 0]);
