<?php
// admin/ids_queue.php — list learners with captured (encrypted) SA IDs that are not yet verified
ini_set('display_errors',1); error_reporting(E_ALL);

require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/db.php';
require_once __DIR__ . '/../lib/helpers.php';
require_once __DIR__ . '/../lib/csrf.php';

require_login();
if (!can('admin')) { http_response_code(403); echo "Admins only."; exit; }

// --- Fetch learners with ID captured but NOT verified ---
// IMPORTANT: Only check for NULL (strict-safe). Do NOT compare to '0000-00-00 00:00:00'.
$rows = [];
$err  = '';
try {
  $sql = "
    SELECT id, name, email, national_id_last4, national_id_doc_path, national_id_verified_at
    FROM users
    WHERE role='learner'
      AND national_id_enc IS NOT NULL
      AND national_id_verified_at IS NULL
    ORDER BY id DESC
  ";
  $st = db()->query($sql);
  $rows = $st->fetchAll();
} catch (Throwable $e) {
  $err = "Could not load queue. If you recently upgraded MySQL, you likely have legacy zero-dates.
          Please run these once in phpMyAdmin:
          SET sql_mode=''; UPDATE users SET national_id_verified_at=NULL
          WHERE national_id_verified_at='0000-00-00 00:00:00';
          Then ensure: ALTER TABLE users MODIFY national_id_verified_at DATETIME NULL DEFAULT NULL;";
}

include __DIR__ . '/../templates/header.php';
?>
<h1>Unverified Learner IDs</h1>
<p class="small">Learners listed here have saved an ID number but are not yet marked verified.</p>

<?php if ($err): ?>
  <div class="alert alert-error"><?= e($err) ?></div>
<?php endif; ?>

<?php if (!$rows): ?>
  <div class="alert alert-ok">Great! No pending ID verifications.</div>
<?php else: ?>
  <table class="mini">
    <tr>
      <th>ID</th>
      <th>Name</th>
      <th>Email</th>
      <th>ID Last4</th>
      <th>Certified ID Doc</th>
      <th>Action</th>
    </tr>
    <?php foreach ($rows as $r): ?>
      <tr>
        <td><?= (int)$r['id'] ?></td>
        <td><?= e($r['name']) ?></td>
        <td><?= e($r['email']) ?></td>
        <td><?= e($r['national_id_last4'] ?: '—') ?></td>
        <td>
          <?php if (!empty($r['national_id_doc_path'])): ?>
            <a class="btn btn-small" target="_blank" href="<?= e(url($r['national_id_doc_path'])) ?>">Open</a>
          <?php else: ?>
            <span class="small muted">No file</span>
          <?php endif; ?>
        </td>
        <td>
          <form method="post" action="<?= e(url('admin/verify_id.php')) ?>" style="display:inline">
            <?php csrf_field(); ?>
            <input type="hidden" name="user_id" value="<?= (int)$r['id'] ?>">
            <button class="btn btn-small">Mark Verified</button>
          </form>
        </td>
      </tr>
    <?php endforeach; ?>
  </table>
<?php endif; ?>

<?php include __DIR__ . '/../templates/footer.php'; ?>
