<?php
// certificate_save_pdf.php — Save a permanent PDF copy of a certificate
// Accepts: POST code=XXXX  or GET ?code=XXXX

ini_set('display_errors',1); error_reporting(E_ALL);

require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';
require_once __DIR__ . '/lib/settings.php';

require_login();
// allow admin/assessor/report roles to save PDFs
if (!(can('admin') || can('grade') || can('report'))) {
  http_response_code(403);
  echo "Not allowed.";
  exit;
}
require_once __DIR__ . '/lib/demo.php';
demo_guard(); // blocks POST in demo mode

// 1) Read code from POST or GET (both supported)
$code = trim($_REQUEST['code'] ?? '');

// 2) If no code, show friendly guidance and exit
if ($code === '') {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Save Certificate PDF</h1>";
  echo "<div class='alert alert-error'>Missing certificate code.</div>";
  echo "<p class='small'>Open <a href='".e(url("certificates_list.php"))."'>Certificates</a> and click <em>Save PDF Copy</em>, or supply the code like:</p>";
  echo "<pre class='small'>".e(url("certificate_save_pdf.php?code=Ca19c5718"))."</pre>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

// 3) Lookup the certificate
$st = db()->prepare("
  SELECT c.*, u.name AS learner_name, u.email, co.title AS course_title
  FROM certificates c
  LEFT JOIN users u ON u.id=c.user_id
  LEFT JOIN courses co ON co.id=c.course_id
  WHERE c.cert_code=? LIMIT 1
");
$st->execute([$code]);
$cert = $st->fetch();

if (!$cert) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Save Certificate PDF</h1>";
  echo "<div class='alert alert-error'>No certificate found for code <strong>".e($code)."</strong>.</div>";
  echo "<p class='small'>Check the code and try again from <a href='".e(url("certificates_list.php"))."'>Certificates</a>.</p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

// 4) Make sure storage folder exists (we use lmis/storage/certs/)
$root = realpath(__DIR__);
$storeRel = 'storage/certs';                    // IMPORTANT: no leading slash
$storeAbs = $root . '/' . $storeRel;           // absolute path on disk
if (!is_dir($storeAbs)) {
  @mkdir($storeAbs, 0775, true);
}

// 5) Render the PDF by calling your existing certificate_pdf endpoint
$pdfUrl = full_url('certificate_pdf.php?code=' . urlencode($code));

// Prefer cURL (more reliable on some hosts); fallback to file_get_contents
function fetch_bytes($url){
  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_SSL_VERIFYPEER => true,
      CURLOPT_SSL_VERIFYHOST => 2,
      CURLOPT_HEADER => false,
      CURLOPT_TIMEOUT => 30,
      CURLOPT_USERAGENT => 'AIT-LMIS/1.0',
    ]);
    $data = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($http === 200 && $data) return $data;
  }
  return @file_get_contents($url);
}

$pdfBin = fetch_bytes($pdfUrl);

// Error check
if ($pdfBin === false || strlen($pdfBin) < 1000) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Save Certificate PDF</h1>";
  echo "<div class='alert alert-error'>Could not generate the PDF for <strong>".e($code)."</strong>.</div>";
  echo "<p class='small'>Try the live render first: <a href='".e(url("certificate_pdf.php?code=".urlencode($code)))."' target='_blank'>Open PDF</a></p>";
  echo "<p class='small'>If that works but saving fails, your host may block internal HTTP calls. I can switch this to render in-process with Dompdf; say the word.</p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

// 6) Save file and update DB
$filename = 'certificate_' . preg_replace('/[^A-Za-z0-9_\-]/','', $code) . '.pdf';
$savePath = $storeAbs . '/' . $filename;        // disk path
$saveRel  = $storeRel . '/' . $filename;        // web path (relative to /lmis)

// write file
if (@file_put_contents($savePath, $pdfBin) === false) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Save Certificate PDF</h1>";
  echo "<div class='alert alert-error'>Server could not write the file to ".e($savePath)."</div>";
  echo "<p class='small'>Check folder permissions (0775) on <code>".e($storeAbs)."</code>.</p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

// update certificate record
try {
  $u = db()->prepare("UPDATE certificates SET pdf_path=?, pdf_saved_at=NOW() WHERE id=?");
  $u->execute([$saveRel, $cert['id']]);
} catch (Throwable $e) { /* not fatal */ }

// 7) Done — show success UI
include __DIR__ . '/templates/header.php';
?>
<h1>PDF Saved</h1>
<div class="alert alert-ok">
  Permanent copy saved for <strong><?= e($cert['learner_name'] ?? 'Learner') ?></strong>,
  course <strong><?= e($cert['course_title'] ?? 'Course') ?></strong>.<br>
  Code: <code><?= e($code) ?></code>
</div>
<p>
  <!-- IMPORTANT: wrap with url() so it becomes /lmis/storage/certs/... -->
  <a class="btn" href="<?= e(url($saveRel)) ?>" target="_blank">Open Saved PDF</a>
  <a class="btn btn-outline" href="<?= e(url('certificates_list.php')) ?>">Back to Certificates</a>
</p>
<?php include __DIR__ . '/templates/footer.php'; ?>
