<?php
// /lmis/certificate_view.php — diagnostic-friendly certificate viewer
// Copy–paste whole file. Safe to keep in production if you flip $DEBUG to false.

$DEBUG = true; // turn to false when done debugging
if ($DEBUG) { ini_set('display_errors', 1); error_reporting(E_ALL); }

require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';
require_once __DIR__ . '/lib/settings.php';
require_once __DIR__ . '/lib/pii.php'; // ok if present; not required for view

// 1) Read ?code=
$code = trim($_GET['code'] ?? '');
if ($code === '') {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Certificate</h1><div class='alert alert-error'>Missing certificate code (?code=...)</div>";
  include __DIR__ . '/templates/footer.php'; exit;
}

// 2) Load certificate
try {
  $st = db()->prepare("SELECT * FROM certificates WHERE cert_code=? LIMIT 1");
  $st->execute([$code]);
  $CERT = $st->fetch();
} catch (Throwable $e) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Certificate</h1><div class='alert alert-error'>DB error loading certificate: ".e($e->getMessage())."</div>";
  include __DIR__ . '/templates/footer.php'; exit;
}

if (!$CERT) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Certificate</h1><div class='alert alert-error'>No certificate found for code <strong>".e($code)."</strong>.</div>";
  include __DIR__ . '/templates/footer.php'; exit;
}

// 3) Load learner & course rows the template expects
try {
  $st = db()->prepare("SELECT * FROM users WHERE id=? LIMIT 1");
  $st->execute([$CERT['user_id']]);
  $LEARN = $st->fetch() ?: [];
} catch (Throwable $e) { $LEARN = []; }

try {
  $st = db()->prepare("SELECT * FROM courses WHERE id=? LIMIT 1");
  $st->execute([$CERT['course_id']]);
  $COURSE = $st->fetch() ?: [];
} catch (Throwable $e) { $COURSE = []; }

// 4) Render page chrome (header)
include __DIR__ . '/templates/header.php';

// 5) Friendly preflight checks (so you see *why* it’s blank)
$problems = [];
if (empty($LEARN))  $problems[] = "Learner record missing for user_id=".e($CERT['user_id']);
if (empty($COURSE)) $problems[] = "Course record missing for course_id=".e($CERT['course_id']);

$templatePath = __DIR__ . '/templates/certificate_render.php';
if (!is_file($templatePath)) {
  $problems[] = "Template missing: <code>".e($templatePath)."</code>";
}

if ($problems) {
  echo "<h1>Certificate</h1>";
  echo "<div class='alert alert-error'><strong>Cannot render certificate</strong><br>";
  echo "<ul class='bullets'>";
  foreach ($problems as $p) echo "<li>". $p ."</li>";
  echo "</ul>";
  echo "<div class='small muted'>Fix the above, then refresh this page.</div>";
  echo "</div>";

  if ($DEBUG) {
    echo "<details style='margin-top:12px'><summary>Debug dump</summary><pre>";
    echo "CERT:\n"; var_dump($CERT);
    echo "\nLEARN:\n"; var_dump($LEARN);
    echo "\nCOURSE:\n"; var_dump($COURSE);
    echo "</pre></details>";
  }

  include __DIR__ . '/templates/footer.php'; exit;
}

// 6) All good — include the same render used by PDF
// The template expects $CERT, $LEARN, $COURSE, and may also use $for_pdf flag
$for_pdf = false;

try {
  include $templatePath;
} catch (Throwable $e) {
  echo "<div class='alert alert-error'>Template error in <code>".e($templatePath)."</code>: ".e($e->getMessage())."</div>";
  if ($DEBUG) {
    echo "<details style='margin-top:12px'><summary>Backtrace</summary><pre>";
    echo e($e->getTraceAsString());
    echo "</pre></details>";
  }
}

// 7) Footer
include __DIR__ . '/templates/footer.php';
