<?php
// company/scores_upload.php — company rep uploads CSV of driver scores/events
ini_set('display_errors',1); error_reporting(E_ALL);
require_once __DIR__.'/../lib/auth.php';
require_once __DIR__.'/../lib/db.php';
require_once __DIR__.'/../lib/helpers.php';
require_once __DIR__.'/../lib/csrf.php';

require_login();
if (!can('company') && !can('admin')) { http_response_code(403); echo "Company reps/admin only."; exit; }

$msg=''; $err='';

if ($_SERVER['REQUEST_METHOD']==='POST') {
  verify_csrf_or_die();

  if (!isset($_FILES['csv']) || $_FILES['csv']['error']!==UPLOAD_ERR_OK) {
    $err = "Please choose a CSV file.";
  } else {
    $tmp = $_FILES['csv']['tmp_name'];
    $fh = fopen($tmp,'r');
    if (!$fh) { $err="Cannot read file."; }
    else {
      // CSV header: email_or_userid, score, category, recorded_at (YYYY-MM-DD HH:MM:SS), source
      // OR events: email_or_userid, event_type, severity_score, event_timestamp, source, metadata(JSON)
      $rows=0; $insScores=0; $insEvents=0;
      while (($line=fgetcsv($fh))!==false) {
        $rows++;
        if ($rows==1 && (stripos($line[0],'email')!==false || stripos($line[0],'user')!==false)) {
          // header row — skip
          continue;
        }
        // Detect scores vs events by number of fields
        if (count($line)>=5 && is_numeric($line[1]) && (int)$line[1] <= 100) {
          // score row
          list($userKey,$score,$category,$when,$source) = $line;
          $uid = resolve_user_id($userKey);
          if ($uid) {
            $st = db()->prepare("INSERT INTO driver_scores(user_id,score,category,source,recorded_at) VALUES(?,?,?,?,?)");
            $st->execute([$uid,(int)$score,trim($category?:'weekly'),trim($source?:'csv'),trim($when?:date('Y-m-d H:i:s'))]);
            $insScores++;
          }
        } elseif (count($line)>=5) {
          // event row
          list($userKey,$etype,$sev,$when,$source) = $line;
          $meta = $line[5] ?? null;
          $uid = resolve_user_id($userKey);
          if ($uid) {
            $st = db()->prepare("INSERT INTO driver_events(user_id,event_type,severity_score,event_timestamp,source,metadata) VALUES(?,?,?,?,?,?)");
            $st->execute([$uid,trim($etype),(int)$sev,trim($when),trim($source?:'csv'), $meta ?: null]);
            $insEvents++;
          }
        }
      }
      fclose($fh);
      $msg = "Uploaded: rows=$rows | scores=$insScores | events=$insEvents.";
    }
  }
}

function resolve_user_id($emailOrId){
  if (ctype_digit((string)$emailOrId)) {
    return (int)$emailOrId;
  }
  $st=db()->prepare("SELECT id FROM users WHERE email=? LIMIT 1");
  $st->execute([trim($emailOrId)]);
  $r=$st->fetch();
  return $r ? (int)$r['id'] : 0;
}

include __DIR__.'/../templates/header.php';
?>
<h1>Upload driver data (CSV)</h1>
<?php if ($msg): ?><div class="alert alert-ok"><?= e($msg) ?></div><?php endif; ?>
<?php if ($err): ?><div class="alert alert-error"><?= e($err) ?></div><?php endif; ?>

<div class="panel"><div class="panel__body">
  <p class="small">CSV formats supported:</p>
  <ul class="bullets small">
    <li><code>email_or_userid, score, category, recorded_at, source</code></li>
    <li><code>email_or_userid, event_type, severity_score, event_timestamp, source, metadata-json</code></li>
  </ul>
</div></div>

<form method="post" enctype="multipart/form-data" class="form-card">
  <?php csrf_field(); ?>
  <label class="field">
    <span class="field__label">Choose CSV file</span>
    <input class="field__input" type="file" name="csv" accept=".csv" required>
  </label>
  <div class="form__actions">
    <button class="btn btn-primary">Upload</button>
    <a class="btn btn-outline" href="<?= e(url('company/dashboard.php')) ?>">Back</a>
  </div>
</form>
<?php include __DIR__.'/../templates/footer.php'; ?>
