<?php
// course.php — learner-facing study page with AUTO-SAVE progress (DRY/KISS)
require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/helpers.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/tracking.php';

require_login();

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) {
  include __DIR__ . '/templates/header.php';
  echo "<p>Course ID missing.</p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

/* Load course */
$s = db()->prepare("SELECT id, title, description FROM courses WHERE id=? LIMIT 1");
$s->execute([$id]);
$course = $s->fetch();
if (!$course) {
  include __DIR__ . '/templates/header.php';
  echo "<p>Course not found.</p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

/* Load modules (ordered) */
$m = db()->prepare("SELECT id, title, html_content FROM modules WHERE course_id=? ORDER BY position ASC, id ASC");
$m->execute([$id]);
$modules = $m->fetchAll();

include __DIR__ . '/templates/header.php';
?>

<h2><?= e($course['title']) ?></h2>
<p><?= nl2br(e((string)$course['description'])) ?></p>

<!-- Lesson content wrapper: the scroll box we measure -->
<div id="lesson-content">
  <ol style="padding-left:18px;">
    <?php foreach($modules as $mod): ?>
      <li class="lesson-block" id="mod-<?= (int)$mod['id'] ?>" data-mod-id="<?= (int)$mod['id'] ?>" style="margin-bottom:18px;">
        <strong><?= e($mod['title']) ?></strong>
        <div style="margin-top:6px;">
          <?= nl2br(e((string)$mod['html_content'])) ?>
        </div>
      </li>
    <?php endforeach; ?>
    <?php if (!$modules): ?>
      <li class="lesson-block" data-mod-id="" style="margin-bottom:18px;">
        <em>No modules yet.</em>
      </li>
    <?php endif; ?>
  </ol>
</div>

<!-- AUTO-SAVE PROGRESS -->
<script>
(function(){
  const COURSE_ID = <?= (int)$course['id'] ?>;
  // collect module blocks
  const blocks = Array.from(document.querySelectorAll('.lesson-block'));
  const contentBox = document.getElementById('lesson-content') || document.body;

  // Get how far we are: percentage of modules whose midpoint is above viewport bottom.
  function calcCourseProgress(){
    if (!blocks.length) return 100;
    const viewportBottom = window.scrollY + window.innerHeight;
    let completedCount = 0;
    let lastModId = null;

    for (const el of blocks){
      const rect = el.getBoundingClientRect();
      const midY = window.scrollY + rect.top + (rect.height/2);
      if (midY <= viewportBottom) {
        completedCount++;
        lastModId = el.dataset.modId ? parseInt(el.dataset.modId, 10) : lastModId;
      }
    }

    const pct = Math.round((completedCount / blocks.length) * 100);
    return { pct: Math.max(0, Math.min(100, pct)), lastModId };
  }

  function saveProgress(){
    if (!COURSE_ID) return;
    const { pct, lastModId } = calcCourseProgress();

    const fd = new FormData();
    fd.append('course_id', COURSE_ID);
    if (lastModId) fd.append('module_id', lastModId);
    fd.append('progress_pct', pct);
    fd.append('last_position_sec', 0); // no video on this page
    fd.append('last_page', location.pathname + location.search);

    fetch('<?= e(url("api/progress_save.php")) ?>', { method:'POST', body: fd })
      .then(r=>r.json())
      .catch(()=>{ /* ignore transient errors */ });

    try {
      localStorage.setItem('progress:'+COURSE_ID, JSON.stringify({t: Date.now(), p: pct}));
    } catch(e) {}
  }

  // Save now, then every 20s
  saveProgress();
  setInterval(saveProgress, 20000);

  // Save when leaving the page
  window.addEventListener('beforeunload', saveProgress);

  // Save after scroll pause (throttle to every ~1.5s)
  let scrollTimer;
  window.addEventListener('scroll', function(){
    if (scrollTimer) return;
    scrollTimer = setTimeout(()=>{ scrollTimer=null; saveProgress(); }, 1500);
  }, {passive:true});
})();
</script>

<?php include __DIR__ . '/templates/footer.php'; ?>
