<?php
// course_view.php — learner-facing course page with auto-save progress (DRY/KISS)
require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';

$id = (int)($_GET['id'] ?? 0);
if ($id <= 0) redirect('courses.php');

/* ========== Load course ========== */
$st = db()->prepare("
  SELECT id, title, description, duration_days, type,
         nqf_level, credits, course_number, expiry_months,
         price, thumbnail_url
  FROM courses WHERE id=? LIMIT 1
");
$st->execute([$id]);
$c = $st->fetch();
if (!$c) redirect('courses.php');

/* ========== Categories (optional) ========== */
$catnames = [];
try {
  $cats = db()->prepare("
    SELECT cc.name
    FROM course_category_map m
    JOIN course_categories cc ON cc.id=m.category_id
    WHERE m.course_id=? ORDER BY cc.name
  ");
  $cats->execute([$id]);
  $catnames = array_column($cats->fetchAll(), 'name');
} catch (Throwable $e) {
  $catnames = []; // tables might not exist yet; safe fallback
}

/* ========== Unit Standards (optional) ========== */
$units = []; $total_credits = 0;
try {
  $us = db()->prepare("
    SELECT us.us_no, us.title, us.credits
    FROM course_unit_standards cus
    JOIN unit_standards us ON us.id=cus.unit_standard_id
    WHERE cus.course_id=? ORDER BY us.us_no
  ");
  $us->execute([$id]);
  $units = $us->fetchAll();
  foreach($units as $u){ $total_credits += (int)$u['credits']; }
} catch (Throwable $e) {
  $units = [];
  // fallback to course.credits if present
  $total_credits = isset($c['credits']) ? (int)$c['credits'] : 0;
}

include __DIR__ . '/templates/header.php';
?>
<h1><?= e($c['title']) ?></h1>

<div class="grid-2" style="align-items:start;">
  <!-- LEFT: Overview + lesson content -->
  <div>
    <?php
      $img = trim((string)($c['thumbnail_url'] ?? ''));
      if ($img === '') $img = url('assets/img/course-placeholder.jpg'); // fallback placeholder
    ?>
    <img src="<?= e($img) ?>" alt="Course thumbnail"
         style="max-width:100%; border:1px solid #e5e7eb; border-radius:8px; margin-bottom:10px;">

    <!-- Short description -->
    <p><?= nl2br(e((string)($c['description'] ?? ''))) ?></p>

    <!-- Chips row -->
    <div class="chips" style="margin-top:8px;">
      <?php if (!empty($c['type'])): ?>
        <span class="chip"><?= e($c['type']) ?></span>
      <?php endif; ?>
      <?php if (!empty($c['nqf_level'])): ?>
        <span class="chip">NQF L<?= (int)$c['nqf_level'] ?></span>
      <?php endif; ?>
      <?php if ($total_credits > 0): ?>
        <span class="chip"><?= (int)$total_credits ?> Credits</span>
      <?php endif; ?>
      <?php if (!empty($c['duration_days'])): ?>
        <span class="chip"><?= (int)$c['duration_days'] ?> days</span>
      <?php endif; ?>
      <?php if (!empty($c['expiry_months'])): ?>
        <span class="chip">Expires <?= (int)$c['expiry_months'] ?> mo</span>
      <?php endif; ?>
      <?php if (!empty($c['course_number'])): ?>
        <span class="chip">Code <?= e($c['course_number']) ?></span>
      <?php endif; ?>
      <?php if (!is_null($c['price'])): ?>
        <span class="chip">R <?= number_format((float)$c['price'],2) ?></span>
      <?php endif; ?>
    </div>

    <!-- Enroll / Login buttons -->
    <div class="form__actions" style="margin-top:12px;">
      <?php if (is_logged_in()): ?>
        <a class="btn btn-primary" href="<?= e(url('enroll.php?course_id='.(int)$id)) ?>">Enroll Now</a>
      <?php else: ?>
        <a class="btn btn-primary" href="<?= e(url('login.php')) ?>">Login to Enroll</a>
        <a class="btn btn-outline" href="<?= e(url('register.php')) ?>">Create Account</a>
      <?php endif; ?>
    </div>

    <!-- ======== BEGIN: Lesson content wrapper (for auto-save scroll) ======== -->
    <div id="lesson-content" style="margin-top:16px;">
      <!-- Put your lesson materials inside this box.
           The auto-save script measures progress in this box.
           Example content: -->
      <!--
      <video id="player" controls style="max-width:100%;">
        <source src="<?= e(url('storage/videos/lesson1.mp4')) ?>" type="video/mp4">
      </video>
      <div style="margin-top:10px;">
        <h3>Lesson 1: Safety Basics</h3>
        <p>… your lesson text, images, downloads …</p>
      </div>
      -->
    </div>
    <!-- ======== END: Lesson content wrapper ======== -->
  </div>

  <!-- RIGHT: Categories + Unit Standards -->
  <div>
    <!-- Categories -->
    <div class="panel">
      <div class="panel__body">
        <h3>Categories</h3>
        <p><?= $catnames ? e(implode(', ', $catnames)) : '<span class="muted">No categories.</span>' ?></p>
      </div>
    </div>

    <!-- Unit Standards -->
    <div class="panel">
      <div class="panel__body">
        <h3>Unit Standards</h3>
        <?php if ($units): ?>
          <table class="mini">
            <tr><th>US #</th><th>Title</th><th>Credits</th></tr>
            <?php foreach($units as $u): ?>
              <tr>
                <td><?= e($u['us_no']) ?></td>
                <td><?= e($u['title']) ?></td>
                <td><?= (int)$u['credits'] ?></td>
              </tr>
            <?php endforeach; ?>
            <tr>
              <th colspan="2" style="text-align:right;">Total</th>
              <th><?= (int)$total_credits ?></th>
            </tr>
          </table>
        <?php else: ?>
          <p class="muted">No unit standards linked yet.</p>
        <?php endif; ?>
      </div>
    </div>
  </div>
</div>

<!-- ========== Auto-save progress (no buttons needed) ========== -->
<script>
(function(){
  // Course/Module/Lesson IDs (modules/lessons can be null for now)
  const COURSE_ID = <?= isset($c['id']) ? (int)$c['id'] : 0 ?>;
  const MODULE_ID = null;
  const LESSON_ID = null;

  // Progress by scroll inside #lesson-content (fallback to window)
  function scrollProgress(){
    const box = document.getElementById('lesson-content') || document.body;
    const h = box.scrollHeight - box.clientHeight;
    if (h <= 0) return 100;
    const y = (box === document.body) ? (window.scrollY || 0) : box.scrollTop;
    return Math.max(0, Math.min(100, Math.round((y / h) * 100)));
  }

  // Optional: if there's a <video id="player">, track time-based progress
  const video = document.getElementById('player');
  if (video) {
    video.addEventListener('timeupdate',()=>{ window._videoPos = video.currentTime; });
  }

  function calcProgress(){
    if (video && video.duration > 0) {
      const p = Math.round((video.currentTime / video.duration) * 100);
      return Math.max(0, Math.min(100, p));
    }
    return scrollProgress();
  }

  function saveProgress(){
    if (!COURSE_ID) return;
    const fd = new FormData();
    fd.append('course_id', COURSE_ID);
    if (MODULE_ID !== null) fd.append('module_id', MODULE_ID);
    if (LESSON_ID !== null) fd.append('lesson_id', LESSON_ID);
    const pct = calcProgress();
    fd.append('progress_pct', pct);
    fd.append('last_position_sec', Math.floor(window._videoPos || 0));
    fd.append('last_page', location.pathname + location.search);

    fetch('<?= e(url("api/progress_save.php")) ?>', { method:'POST', body: fd })
      .then(r=>r.json()).catch(()=>{ /* ignore network hiccups */ });

    // local safety (offline)
    try {
      localStorage.setItem('progress:'+COURSE_ID, JSON.stringify({t: Date.now(), p: pct}));
    } catch(e) {}
  }

  // Fire now, then every 20s, and when leaving
  saveProgress();
  setInterval(saveProgress, 20000);
  window.addEventListener('beforeunload', saveProgress);

  // Also save after user stops scrolling (throttled)
  let t;
  const scrollBox = document.getElementById('lesson-content') || window;
  scrollBox.addEventListener('scroll', function(){
    if (t) return;
    t = setTimeout(()=>{ t=null; saveProgress(); }, 3000);
  }, {passive:true});
})();
</script>

<?php include __DIR__ . '/templates/footer.php'; ?>
