<?php
// dashboard.php — Role-aware AIT LMIS dashboard (DRY/KISS)
// Shows cards/sections per role. Adds learner driver-score recommendation widget.

ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';
require_once __DIR__ . '/lib/settings.php';

require_login();
$u = current_user();

/* ---------- tiny DB helpers ---------- */
function qv($sql, $params = []){ $s = db()->prepare($sql); $s->execute($params); return $s->fetch(); }
function qa($sql, $params = []){ $s = db()->prepare($sql); $s->execute($params); return $s->fetchAll(); }
function qn($sql, $params = []){ try { $r = qv($sql,$params); return (int)array_values($r ?? [0])[0]; } catch(Throwable $e){ return 0; } }
function has_table($table){
  try { db()->query("SELECT 1 FROM `$table` LIMIT 1"); return true; }
  catch(Throwable $e){ return false; }
}
function has_col($table,$col){
  try { $st=db()->prepare("SHOW COLUMNS FROM `$table` LIKE ?"); $st->execute([$col]); return (bool)$st->fetch(); }
  catch(Throwable $e){ return false; }
}

/* ---------- optional driver-score recommender ---------- */
/* This file should define:
     latest_driver_score($user_id, $category='weekly') -> ['score'=>int,...] or null
     recommended_course_for_score($score) -> ['course_id'=>int,'title'=>string] or null
   We fail gracefully if it isn't present yet.
*/
$reco_available = false;
$latest_score_fn = null;
$reco_fn = null;
if (file_exists(__DIR__.'/lib/recommend.php')) {
  require_once __DIR__.'/lib/recommend.php';
  $reco_available = function_exists('latest_driver_score') && function_exists('recommended_course_for_score');
  $latest_score_fn = 'latest_driver_score';
  $reco_fn = 'recommended_course_for_score';
}

/* ---------- existence flags ---------- */
$hasEnroll   = has_table('enrollments');
$hasResults  = has_table('results');
$hasSubs     = has_table('submissions');
$hasModules  = has_table('modules');

/* ---------- common totals (safe) ---------- */
$totalCourses   = qn("SELECT COUNT(*) FROM courses");
$totalLearners  = qn("SELECT COUNT(*) FROM users WHERE role='learner'");
$totalAssessors = qn("SELECT COUNT(*) FROM users WHERE role='assessor'");
$totalCerts     = qn("SELECT COUNT(*) FROM certificates");

$cards=[]; $sections=[]; $quick=[];

/* =========================================================
   LEARNER view
   ========================================================= */
if ($u['role']==='learner') {

  // 7.2 — Driver score widget (added FIRST so it appears near the top)
  if ($reco_available) {
    $ls = $latest_score_fn($u['id'], 'weekly'); // or 'monthly'—your choice
    if ($ls && isset($ls['score'])) {
      $score = (int)$ls['score'];
      $rec   = $reco_fn($score);
      $recHtml = $rec
        ? "<a class='btn btn-small' href='".e(url('course_view.php?id='.$rec['course_id']))."'>Start: ".e($rec['title'])."</a>"
        : "<span class='small muted'>No recommendation yet.</span>";

      $sections[] = [
        'title'=>'Your Driving Safety Score',
        'body'=>"
          <p>Latest weekly score: <strong>{$score}</strong></p>
          {$recHtml}
        ",
      ];
    }
  }

  // "My courses" source:
  // 1) enrollments if table exists; else
  // 2) distinct courses from submissions; else 0.
  $myCourses = [];
  if ($hasEnroll){
    $myCourses = qa("SELECT c.id, c.title
                      FROM enrollments e JOIN courses c ON c.id=e.course_id
                     WHERE e.user_id=? ORDER BY c.title", [$u['id']]);
  } elseif ($hasSubs){
    $myCourses = qa("SELECT DISTINCT c.id, c.title
                       FROM submissions s JOIN courses c ON c.id=s.course_id
                      WHERE s.user_id=? ORDER BY c.title", [$u['id']]);
  }

  // Build progress cards using modules/submissions if available
  foreach ($myCourses as $mc){
    $mods = ($hasModules) ? qn("SELECT COUNT(*) FROM modules WHERE course_id=?", [$mc['id']]) : 0;
    $subm = ($hasSubs)    ? qn("SELECT COUNT(*) FROM submissions WHERE user_id=? AND course_id=?", [$u['id'],$mc['id']]) : 0;
    $pct  = ($mods>0) ? min(100, round(($subm/$mods)*100)) : (($subm>0)? 50 : 0);

    $sections[] = [
      'title'=>$mc['title'],
      'body'=>"
        <div class='progress'><div class='progress__bar' style='width:{$pct}%'></div></div>
        <div class='small'>Progress: <strong>{$pct}%</strong> ".($mods? "· {$subm}/{$mods} submissions":"")."</div>
        <div class='spacer8'></div>
        <a class='btn btn-small' href='".e(url('course_view.php?id='.$mc['id']))."'>Open course</a>
        <a class='btn btn-small btn-outline' href='".e(url('upload_poe.php?course_id='.$mc['id']))."'>Upload POE</a>
      ",
    ];
  }

  $cards = [
    ['title'=>'My Courses','value'=>count($myCourses),'hint'=>'Enrolled/active','link'=>url('courses.php')],
    ['title'=>'Certificates','value'=>qn("SELECT COUNT(*) FROM certificates WHERE user_id=?",[$u['id']]),'hint'=>'Issued to you','link'=>url('my_certs.php')],
    ['title'=>'AI Tutor','value'=>'Ask','hint'=>'Instant help','link'=>url('ai_tutor.php')],
  ];

  $quick = [
    ['label'=>'View Courses','href'=>url('courses.php')],
    ['label'=>'Upload POE','href'=>url('upload_poe.php')],
    ['label'=>'My Certificates','href'=>url('my_certs.php')],
    ['label'=>'Ask AI Tutor','href'=>url('ai_tutor.php')],
  ];

  if (empty($myCourses)) {
    $sections[] = [
      'title'=>'Get started',
      'body'=>"<p>You have no active courses yet.</p>
               <a class='btn btn-small' href='".e(url('courses.php'))."'>Browse courses</a>"
    ];
  }
}

/* =========================================================
   ASSESSOR / ADMIN view
   ========================================================= */
if ($u['role']==='assessor' || $u['role']==='admin') {

  // Pending POEs:
  if ($hasSubs){
    if (has_col('submissions','status')) {
      $pendingPOE = qn("SELECT COUNT(*) FROM submissions WHERE status IN ('submitted','needs_review')");
    } else {
      $pendingPOE = qn("SELECT COUNT(*) FROM submissions");
    }
  } else $pendingPOE = 0;

  // NYC to verify (only if results exists)
  if ($hasResults){
    $toVerifyNYC = has_col('results','verified_at')
      ? qn("SELECT COUNT(*) FROM results WHERE decision='NYC' AND verified_at IS NULL")
      : qn("SELECT COUNT(*) FROM results WHERE decision='NYC'");
  } else $toVerifyNYC = 0;

  // Ready to issue (requires results)
  if ($hasResults){
    if (has_col('results','verified_at')){
      $toIssueCerts = qn("
        SELECT COUNT(*) FROM results r
         WHERE r.decision='C' AND r.verified_at IS NOT NULL
           AND NOT EXISTS(SELECT 1 FROM certificates c WHERE c.user_id=r.user_id AND c.course_id=r.course_id)
      ");
    } else {
      $toIssueCerts = qn("
        SELECT COUNT(*) FROM results r
         WHERE r.decision='C'
           AND NOT EXISTS(SELECT 1 FROM certificates c WHERE c.user_id=r.user_id AND c.course_id=r.course_id)
      ");
    }
  } else $toIssueCerts = 0;

  $cards = [
    ['title'=>'Pending POEs','value'=>$pendingPOE,'hint'=>$hasSubs?'Need grading':'No submissions table','link'=> $hasSubs? url('results.php?tab=pending') : url('reports.php')],
    ['title'=>'Verify NYC','value'=>$toVerifyNYC,'hint'=>$hasResults?'Check borderline':'No results table','link'=> $hasResults? url('results.php?tab=nyc') : url('reports.php')],
    ['title'=>'Issue Certificates','value'=>$toIssueCerts,'hint'=>$hasResults?'Ready to issue':'No results table','link'=> $hasResults? url('results.php?tab=issue') : url('certificates_list.php')],
  ];

  // Recent submissions (only if table exists)
  $recent = [];
  if ($hasSubs){
    $recent = qa("
      SELECT s.id, u.name AS learner, c.title AS course, DATE_FORMAT(s.created_at,'%Y-%m-%d') AS d
        FROM submissions s
        JOIN users u ON u.id=s.user_id
        JOIN courses c ON c.id=s.course_id
       ORDER BY s.created_at DESC
       LIMIT 6
    ");
  }
  $html = "<table class='mini'><tr><th>Learner</th><th>Course</th><th>Date</th><th></th></tr>";
  foreach ($recent as $r){
    $html .= "<tr>
      <td>".e($r['learner'])."</td>
      <td>".e($r['course'])."</td>
      <td>".e($r['d'])."</td>
      <td><a class='btn btn-small' href='".e(url('grade.php?id='.$r['id']))."'>Grade</a></td>
    </tr>";
  }
  $html .= "</table>";
  $sections[] = [
    'title'=>'Recent submissions',
    'body'=> $recent ? $html : '<p class="small">No recent submissions (or table missing).</p>',
  ];

  $quick = [
    ['label'=>'Open Grading Queue','href'=> $hasSubs? url('results.php?tab=pending') : url('reports.php')],
    ['label'=>'AI Draft Feedback','href'=>url('ai_feedback.php')],
    ['label'=>'Certificates','href'=>url('certificates_list.php')],
    ['label'=>'Reports','href'=>url('reports.php')],
  ];
}

/* =========================================================
   MODERATOR / ADMIN
   ========================================================= */
if ($u['role']==='moderator' || $u['role']==='admin') {
  $toSample = $hasResults
    ? (has_col('results','verified_at') && has_col('results','moderated_at')
        ? qn("SELECT COUNT(*) FROM results WHERE verified_at IS NOT NULL AND moderated_at IS NULL")
        : qn("SELECT COUNT(*) FROM results"))
    : 0;
  $locked = ($hasResults && has_col('results','moderated_at')) ? qn("SELECT COUNT(*) FROM results WHERE moderated_at IS NOT NULL") : 0;

  $sections[] = [
    'title'=>'Moderation queue',
    'body'=>"<p>To sample/lock: <strong>{$toSample}</strong><br>Already locked: <strong>{$locked}</strong></p>
             <a class='btn btn-small' href='".e(url('moderation.php'))."'>Open moderation</a>",
  ];
}

/* =========================================================
   TETA Read-only
   ========================================================= */
if ($u['role']==='teta_readonly') {
  $cards = [
    ['title'=>'Courses','value'=>$totalCourses,'hint'=>'Registered courses','link'=>url('courses.php')],
    ['title'=>'Learners','value'=>$totalLearners,'hint'=>'Active learners','link'=>url('reports.php?view=learners')],
    ['title'=>'Certificates','value'=>$totalCerts,'hint'=>'Issued to date','link'=>url('certificates_list.php')],
  ];
  $sections[] = [
    'title'=>'Compliance snapshots',
    'body'=>"<ul class='bullets'>
      <li>Role-based access (learner/assessor/moderator/TETA)</li>
      <li>QR + hash on certificates (verify online)</li>
      <li>POE uploads + audit trail on grading</li>
      <li>Exports: CSV / Excel / PDF</li>
    </ul>",
  ];
  $quick = [
    ['label'=>'Open Reports','href'=>url('reports.php')],
    ['label'=>'Certificates','href'=>url('certificates_list.php')],
    ['label'=>'Verify a Certificate','href'=>url('verify.php')],
  ];
}

/* =========================================================
   ADMIN extras
   ========================================================= */
if ($u['role']==='admin') {
  $sections[]=[
    'title'=>'Admin quick links',
    'body'=>"<div class='chips'>
      <a class='chip' href='".e(url('courses.php'))."'>Manage Courses</a>
      <a class='chip' href='".e(url('users.php'))."'>Manage Users</a>
      <a class='chip' href='".e(url('settings_provider.php'))."'>Branding & Settings</a>
      <a class='chip' href='".e(url('reports.php'))."'>Reports</a>
    </div>"
  ];
}

/* ---------- render ---------- */
include __DIR__ . '/templates/header.php';
?>
<h1 style="margin-bottom:6px;">Hello, <?= e($u['name']) ?></h1>
<p class="small" style="margin-top:0;">Your <?= e(strtoupper($u['role'])) ?> dashboard</p>

<div class="dash-cards">
  <?php if (empty($cards)): ?>
    <div class="card">
      <h3>Welcome</h3>
      <p>Use the navigation above to get started.</p>
    </div>
  <?php else: foreach ($cards as $c): ?>
    <article class="card card--stat">
      <div class="stat__value"><?= e($c['value']) ?></div>
      <div class="stat__label"><?= e($c['title']) ?></div>
      <div class="small muted"><?= e($c['hint'] ?? '') ?></div>
      <?php if (!empty($c['link'])): ?>
        <div class="spacer8"></div>
        <a class="btn btn-small" href="<?= e($c['link']) ?>">Open</a>
      <?php endif; ?>
    </article>
  <?php endforeach; endif; ?>
</div>

<?php foreach ($sections as $s): ?>
  <section class="panel">
    <h2 class="panel__title"><?= e($s['title']) ?></h2>
    <div class="panel__body"><?= $s['body'] ?></div>
  </section>
<?php endforeach; ?>

<?php if (!empty($quick)): ?>
  <section class="panel">
    <h2 class="panel__title">Quick actions</h2>
    <div class="chips">
      <?php foreach ($quick as $q): ?>
        <a class="chip" href="<?= e($q['href']) ?>"><?= e($q['label']) ?></a>
      <?php endforeach; ?>
    </div>
  </section>
<?php endif; ?>

<?php include __DIR__ . '/templates/footer.php'; ?>
