<?php
// grade.php — view one submission and record C/NYC (assessor/admin)
require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';

require_login();
if (!(can('grade') || can('admin'))) {
  http_response_code(403); echo "Forbidden"; exit;
}

// ---------- helpers ----------
function qa($sql,$p=[]){ $s=db()->prepare($sql); $s->execute($p); return $s->fetchAll(); }
function qv($sql,$p=[]){ $s=db()->prepare($sql); $s->execute($p); return $s->fetch(); }
function has_table($t){ try{ db()->query("SELECT 1 FROM `$t` LIMIT 1"); return true; } catch(Throwable $e){ return false; } }
function has_col($t,$c){
  try{ $st=db()->prepare("SHOW COLUMNS FROM `$t` LIKE ?"); $st->execute([$c]); return (bool)$st->fetch(); }
  catch(Throwable $e){ return false; }
}
function ensure_results_table(){
  $sql = "CREATE TABLE IF NOT EXISTS results (
            id INT NOT NULL AUTO_INCREMENT,
            user_id INT NOT NULL,
            course_id INT NOT NULL,
            assessor_id INT DEFAULT NULL,
            decision ENUM('C','NYC') NOT NULL,
            feedback TEXT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            verified_at DATETIME NULL,
            moderated_at DATETIME NULL,
            PRIMARY KEY (id),
            UNIQUE KEY uq_res_user_course (user_id, course_id),
            KEY idx_res_user (user_id),
            KEY idx_res_course (course_id),
            KEY idx_res_assessor (assessor_id)
          ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
  db()->exec($sql);
}
ensure_results_table();

// ---------- load submission ----------
$subId = (int)($_GET['id'] ?? 0);
$submission = null;
if ($subId > 0 && has_table('submissions')){
  $submission = qv("
  SELECT s.*, u.name AS learner, u.email, u.role AS learner_role, c.title AS course
    FROM submissions s
    JOIN users u   ON u.id = s.user_id
    JOIN courses c ON c.id = s.course_id
   WHERE s.id = ?
     AND u.role = 'learner'
   LIMIT 1
", [$subId]);

if (!$submission) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Grade</h1>";
  echo "<p class='alert alert-error'>This submission is not linked to a learner (likely created by a staff account). Please reassign to a learner.</p>";
  echo "<p><a class='btn' href='".e(url('results.php?tab=pending'))."'>Back to queue</a></p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

}

if (!$submission) {
  include __DIR__ . '/templates/header.php';
  echo "<h1>Grade</h1>";
  echo "<p class='alert alert-error'>Submission not found (id=".e($subId).").</p>";
  echo "<p><a class='btn' href='".e(url('results.php?tab=pending'))."'>Back to queue</a></p>";
  include __DIR__ . '/templates/footer.php';
  exit;
}

// ---------- handle POST (save result) ----------
$msg=''; $err='';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $decision = $_POST['decision'] ?? '';
  $feedback = trim($_POST['feedback'] ?? '');

  if (!in_array($decision, ['C','NYC'], true)) {
    $err = "Please choose a decision (C or NYC).";
  } else {
    // upsert into results
    $sql = "INSERT INTO results (user_id,course_id,assessor_id,decision,feedback,created_at)
            VALUES (?,?,?,?,?,NOW())
            ON DUPLICATE KEY UPDATE
              assessor_id=VALUES(assessor_id),
              decision=VALUES(decision),
              feedback=VALUES(feedback),
              created_at=VALUES(created_at)";
    $st = db()->prepare($sql);
    $st->execute([
      (int)$submission['user_id'],
      (int)$submission['course_id'],
      current_user()['id'],
      $decision,
      $feedback
    ]);

    // If submissions.status exists, mark this submission graded
    if (has_col('submissions','status')) {
      $st2 = db()->prepare("UPDATE submissions SET status='graded' WHERE id=?");
      $st2->execute([$submission['id']]);
    }

    // After save, go back to results queue (pending)
    redirect('results.php?tab=pending');
  }
}

// ---------- render ----------
include __DIR__ . '/templates/header.php';
?>
<h1>Grade Submission</h1>

<section class="panel">
  <h2 class="panel__title">Submission #<?= e($submission['id']) ?></h2>
  <div class="panel__body">
    <p><strong>Learner:</strong> <?= e($submission['learner']) ?> (<?= e($submission['email']) ?>)</p>
    <p><strong>Course:</strong> <?= e($submission['course']) ?></p>
    <?php if (!empty($submission['created_at'])): ?>
      <p><strong>Submitted:</strong> <?= e($submission['created_at']) ?></p>
    <?php endif; ?>

    <?php
      // If you store a file path or URL in e.g. submissions.file_path, show a link
      $fileLink = '';
      foreach (['file_path','file','path','url'] as $col) {
        if (isset($submission[$col]) && $submission[$col]) { $fileLink = $submission[$col]; break; }
      }
      if ($fileLink) {
        // if relative path, build full url
        if (strpos($fileLink,'http') !== 0) $fileLink = url(ltrim($fileLink,'/'));
        echo "<p><a class='btn btn-outline' target='_blank' href='".e($fileLink)."'>Open POE / Attachment</a></p>";
      } else {
        echo "<p class='small muted'>No attachment link found in this submission record.</p>";
      }
    ?>
  </div>
</section>

<form method="post" class="form-card">
  <?php if ($err): ?><div class="alert alert-error"><?= e($err) ?></div><?php endif; ?>

  <fieldset class="field">
    <span class="field__label">Decision</span>
    <label><input type="radio" name="decision" value="C" required> Competent (C)</label>
    &nbsp;&nbsp;
    <label><input type="radio" name="decision" value="NYC" required> Not Yet Competent (NYC)</label>
  </fieldset>

  <label class="field">
    <span class="field__label">Feedback (optional)</span>
    <textarea name="feedback" class="field__input" rows="5" placeholder="Key strengths, evidence gaps, and next steps..."></textarea>
  </label>

  <div class="form__actions">
    <button class="btn btn-primary" type="submit">Save</button>
    <a class="btn btn-outline" href="<?= e(url('results.php?tab=pending')) ?>">Cancel</a>
  </div>
</form>

<?php include __DIR__ . '/templates/footer.php'; ?>
