<?php
// lib/auth.php — sessions, user auth, and simple RBAC
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/helpers.php';

// Harden session cookies
ini_set('session.cookie_httponly', '1');
ini_set('session.use_strict_mode', '1');
ini_set('session.cookie_samesite', 'Lax');
// IMPORTANT: only set '1' on HTTPS
if (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') {
  ini_set('session.cookie_secure', '1');
}

if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

// After successful login, always regenerate:
function login_user($user_id){
  session_regenerate_id(true);
  $_SESSION['uid'] = (int)$user_id;
}


// Start session early and once
if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

// Return current user row (cached per request)
function current_user(){
  static $cached = null;
  if ($cached !== null) return $cached;
  if (!is_logged_in()) return null;

  $uid = (int)($_SESSION['uid'] ?? 0);
  if ($uid <= 0) return null;

  try {
    $s = db()->prepare("SELECT id, name, email, role FROM users WHERE id=? LIMIT 1");
    $s->execute([$uid]);
    $cached = $s->fetch() ?: null;
    return $cached;
  } catch (Throwable $e) {
    return null;
  }
}

function is_logged_in(){ return !empty($_SESSION['uid']); }

function logout_user(){
  $_SESSION = [];
  if (ini_get('session.use_cookies')) {
    $p = session_get_cookie_params();
    setcookie(session_name(), '', time()-42000, $p['path'], $p['domain'], $p['secure'], $p['httponly']);
  }
  session_destroy();
}

function require_login(){
  if (!is_logged_in()) redirect('login.php');
}

/* ---- Roles & Abilities ----
   roles: learner, assessor, moderator, teta_readonly, admin */
function role_abilities($role){
  $map = [
    'learner'       => ['learn','poe_upload'],
    'assessor'      => ['learn','poe_upload','grade','report','cert_issue'],
    'moderator'     => ['learn','poe_upload','moderate','report'],
    'teta_readonly' => ['report_readonly'],
    'admin'         => ['learn','poe_upload','grade','report','cert_issue','moderate','admin'],
  ];
  return $map[$role] ?? [];
}

// SINGLE can() — do NOT redeclare anywhere else
function can($ability){
  $u = current_user();
  if (!$u) return false;
  if ($u['role'] === 'admin') return true;
  return in_array($ability, role_abilities($u['role']), true);
}

function require_ability($ability){
  if (!can($ability)) {
    http_response_code(403);
    echo "<p>Forbidden: you do not have permission to perform this action.</p>";
    exit;
  }
}
