<?php
// lib/cert_bundle.php — single source of truth for certificate data
ini_set('display_errors',1); error_reporting(E_ALL);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/helpers.php';
require_once __DIR__ . '/pii.php';
require_once __DIR__ . '/settings.php';

/**
 * Load certificate + learner + course with consistent logic.
 * - Case-insensitive code lookup (handles cb64.. vs Cb64..)
 * - Learner ID: prefer decrypted national_id_enc; else legacy id_number (only if 13 digits)
 * - Expiry: prefer certificates.expires_at; else derive issued_at + course.expiry_months
 * Returns associative array with CERT, LEARN, COURSE and normalized fields.
 * Throws RuntimeException on errors.
 */
function load_certificate_bundle(string $code): array {
    $code = trim($code);
    if ($code === '') throw new RuntimeException('Missing certificate code.');

    // Case-insensitive lookup
    $st = db()->prepare("SELECT * FROM certificates WHERE LOWER(cert_code)=LOWER(?) LIMIT 1");
    $st->execute([$code]);
    $CERT = $st->fetch();
    if (!$CERT) throw new RuntimeException('Certificate not found.');

    // Learner & course
    $st = db()->prepare("SELECT * FROM users WHERE id=? LIMIT 1");
    $st->execute([$CERT['user_id']]);
    $LEARN = $st->fetch() ?: [];

    $st = db()->prepare("SELECT * FROM courses WHERE id=? LIMIT 1");
    $st->execute([$CERT['course_id']]);
    $COURSE = $st->fetch() ?: [];

    // --- Learner ID (unified rule) ---
    $learner_id = '';
    try {
        if (!empty($LEARN['national_id_enc'])) {
            $learner_id = (string)pii_decrypt($LEARN['national_id_enc']);
        } elseif (!empty($LEARN['id_number'])) {
            $learner_id = (string)$LEARN['id_number'];
        }
    } catch (Throwable $e) { /* ignore */ }
    if (!preg_match('/^\d{13}$/', $learner_id)) {
        $learner_id = ''; // unify "missing" to empty string
    }

    // --- Dates (normalize once) ---
    $issued  = $CERT['issued_at']  ?? null;
    $expires = $CERT['expires_at'] ?? null;

    if ((empty($expires) || $expires === '0000-00-00') && !empty($issued) && !empty($COURSE['expiry_months'])) {
        $dt = new DateTime($issued);
        $dt->modify('+' . (int)$COURSE['expiry_months'] . ' months');
        $expires = $dt->format('Y-m-d');
    }

    $issued_fmt  = (!empty($issued)  && $issued  !== '0000-00-00') ? (new DateTime($issued))->format('Y-m-d')  : '—';
    $expires_fmt = (!empty($expires) && $expires !== '0000-00-00') ? (new DateTime($expires))->format('Y-m-d') : '—';

    // Display code always with C-AIT- prefix (display only)
    $rawCode     = (string)($CERT['cert_code'] ?? '');
    $displayCode = (stripos($rawCode, 'C-AIT-') === 0) ? $rawCode : ('C-AIT-' . $rawCode);

    return [
        'CERT'         => $CERT,
        'LEARN'        => $LEARN,
        'COURSE'       => $COURSE,
        'learner_id'   => $learner_id,
        'issued_fmt'   => $issued_fmt,
        'expires_fmt'  => $expires_fmt,
        'display_code' => $displayCode,
    ];
}
