<?php
// lib/pii.php — DRY/KISS AES-256-GCM helpers for SA ID numbers

$secretPath = __DIR__ . '/secret.php';
if (!is_file($secretPath)) {
  http_response_code(500);
  echo "<h1>Server setup incomplete</h1>
        <p>Missing <code>lib/secret.php</code>. Create it with PII_KEY_B64 and PII_IV_SALT_B64.</p>";
  exit;
}
require_once $secretPath;

function pii_key(): string {
  $key = base64_decode(PII_KEY_B64, true);
  if ($key === false || strlen($key) !== 32) {
    throw new RuntimeException("Bad PII_KEY_B64: must be base64 for 32 raw bytes.");
  }
  return $key;
}

function pii_nonce(): string {
  // Build a 12-byte nonce for GCM: derive from salt + random
  $salt = base64_decode(PII_IV_SALT_B64, true);
  if ($salt === false || strlen($salt) < 4) $salt = random_bytes(8);
  return substr(hash('sha256', $salt . random_bytes(16), true), 0, 12);
}

/**
 * Encrypt plaintext. Returns a compact token: base64(nonce) | base64(tag) | base64(cipher)
 */
function pii_encrypt(string $plaintext): string {
  if ($plaintext === '') return '';
  $key   = pii_key();
  $nonce = pii_nonce();
  $tag   = '';
  $cipher= openssl_encrypt($plaintext, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $nonce, $tag);
  if ($cipher === false) throw new RuntimeException("PII encrypt failed");
  return base64_encode($nonce) . '|' . base64_encode($tag) . '|' . base64_encode($cipher);
}

/**
 * Decrypt the token format above. Returns '' if invalid.
 */
function pii_decrypt(?string $blob): string {
  if (!$blob) return '';
  $parts = explode('|', $blob);
  if (count($parts) !== 3) return '';
  [$nB64, $tB64, $cB64] = $parts;
  $nonce  = base64_decode($nB64, true);
  $tag    = base64_decode($tB64, true);
  $cipher = base64_decode($cB64, true);
  if ($nonce === false || $tag === false || $cipher === false) return '';
  $key = pii_key();
  $plain = openssl_decrypt($cipher, 'aes-256-gcm', $key, OPENSSL_RAW_DATA, $nonce, $tag);
  return $plain === false ? '' : $plain;
}

// Utility: last 4 digits for display/search
function pii_last4(string $id13): string {
  return substr(preg_replace('/\D+/', '', $id13), -4) ?: '';
}
