<?php
// lib/settings.php — fail-safe K/V store (won't fatal if DB/table missing)

require_once __DIR__ . '/db.php';

function __settings_db_or_null() {
    try { return db(); } catch (Throwable $e) { return null; }
}

function __settings_cache(&$store = null) {
    static $cache = [];
    if ($store !== null) { $cache = $store; }
    return $cache;
}

/** Get a setting by key. Never fatal. Falls back to $default on any error. */
function sget(string $key, $default = '') {
    // cached?
    $cache = __settings_cache();
    if (array_key_exists($key, $cache)) return $cache[$key];

    $pdo = __settings_db_or_null();
    if (!$pdo) { $cache[$key] = $default; __settings_cache($cache); return $default; }

    try {
        $st = $pdo->prepare("SELECT s_value FROM settings WHERE s_key=? LIMIT 1");
        $st->execute([$key]);
        $val = $st->fetchColumn();
        if ($val === false) $val = $default;
    } catch (Throwable $e) {
        $val = $default; // table might not exist yet
    }

    $cache[$key] = $val;
    __settings_cache($cache);
    return $val;
}

/** Set a setting. Never fatal — ignores errors if table/DB is not ready. */
function sset(string $key, $value): void {
    $pdo = __settings_db_or_null();
    if (!$pdo) return;
    try {
        $st = $pdo->prepare("
            INSERT INTO settings (s_key, s_value)
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE s_value=VALUES(s_value), updated_at=CURRENT_TIMESTAMP
        ");
        $st->execute([$key, $value]);
        $cache = __settings_cache(); $cache[$key] = $value; __settings_cache($cache);
    } catch (Throwable $e) {
        // swallow — keeps pages loading
    }
}

// Compatibility aliases
function setting(string $key, $default = '') { return sget($key, $default); }
function set_setting(string $key, $value): void { sset($key, $value); }
