<?php
// moderate_action.php — safe POST endpoint for assessor/moderator actions
ini_set('display_errors', 0); // keep off in prod
error_reporting(E_ALL);

require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';
require_once __DIR__ . '/lib/csrf.php';
require_once __DIR__ . '/lib/demo.php';

require_login();
demo_guard(); // blocks POST if DEMO_MODE

// Only assessor, moderator, or admin may act
if (!(can('grade') || can('moderate') || can('admin'))) {
  http_response_code(403);
  echo "Forbidden.";
  exit;
}

// POST only
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo "Method Not Allowed";
  exit;
}

// CSRF
verify_csrf_or_die();

// Tiny helpers
function has_col($table,$col){
  try {
    $st = db()->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
    $st->execute([$col]);
    return (bool)$st->fetch();
  } catch (Throwable $e) {
    return false;
  }
}

$u = current_user();
$result_id = (int)($_POST['result_id'] ?? 0);
$action    = trim($_POST['action'] ?? '');

// Must have a target result
if ($result_id <= 0) {
  redirect('moderation.php?err=bad+result');
}

// Load the result row (and a bit of context for sanity)
$st = db()->prepare("
  SELECT r.*
  FROM results r
  WHERE r.id = ?
  LIMIT 1
");
$st->execute([$result_id]);
$r = $st->fetch();

if (!$r) {
  redirect('moderation.php?err=result+not+found');
}

// We will conditionally update columns only if they exist
$has_verified_at   = has_col('results','verified_at');
$has_moderated_at  = has_col('results','moderated_at');
$has_moderated_by  = has_col('results','moderated_by');
$has_decision      = has_col('results','decision');
$has_decided_at    = has_col('results','decided_at');
$has_decided_by    = has_col('results','decided_by');

// Build update dynamically based on action
$fields = [];
$params = [];

switch ($action) {
  // Mark result verified/unverified (Assessor verification)
  case 'verify':
    if ($has_verified_at) { $fields[] = "verified_at = NOW()"; }
    break;

  case 'unverify':
    if ($has_verified_at) { $fields[] = "verified_at = NULL"; }
    break;

  // Lock via Moderation (final)
  case 'lock':
    if ($has_moderated_at) { $fields[] = "moderated_at = NOW()"; }
    if ($has_moderated_by) { $fields[] = "moderated_by = ?"; $params[] = $u['id']; }
    break;

  // Set decision to Competent or Not Yet Competent
  case 'decision':
    if (!$has_decision) {
      redirect('moderation.php?err=decision+column+missing');
    }
    $val = strtoupper(trim($_POST['value'] ?? ''));
    if (!in_array($val, ['C','NYC'], true)) {
      redirect('moderation.php?err=bad+decision');
    }
    $fields[] = "decision = ?";
    $params[] = $val;
    if ($has_decided_at) { $fields[] = "decided_at = NOW()"; }
    if ($has_decided_by) { $fields[] = "decided_by = ?"; $params[] = $u['id']; }
    break;

  default:
    redirect('moderation.php?err=unknown+action');
}

// If nothing to update (e.g., columns missing), bail gracefully
if (empty($fields)) {
  redirect('moderation.php?msg=nothing+to+update');
}

// Execute update
$sql = "UPDATE results SET ".implode(', ', $fields)." WHERE id = ?";
$params[] = $result_id;

try {
  $up = db()->prepare($sql);
  $up->execute($params);
} catch (Throwable $e) {
  // Log to server error log, show friendly UI
  error_log("moderate_action update failed: ".$e->getMessage());
  redirect('moderation.php?err=update+failed');
}

// Success — bounce back to moderation queue
redirect('moderation.php?msg=ok');
