<?php
// moderation.php — simple moderation queue (verify -> lock) DRY/KISS
require_once __DIR__ . '/lib/auth.php';
require_once __DIR__ . '/lib/db.php';
require_once __DIR__ . '/lib/helpers.php';

require_login();
if (!(can('moderate') || can('admin'))) { http_response_code(403); echo "Moderators/Admins only."; exit; }

include __DIR__ . '/templates/header.php';

// Fetch queue: verified but not yet moderated
// If your 'verified_at' doesn't exist, we fallback to all results with no moderated_at.
$hasVerified = false;
try {
  $st = db()->query("SHOW COLUMNS FROM results LIKE 'verified_at'");
  $hasVerified = (bool)$st->fetch();
} catch (Throwable $e) {}

$sql = "
  SELECT r.id, r.user_id, r.course_id, r.decision,
         r.assessor_id, r.verified_at, r.moderated_at,
         u.name AS learner_name, u.email AS learner_email,
         c.title AS course_title
    FROM results r
    JOIN users u   ON u.id=r.user_id
    JOIN courses c ON c.id=r.course_id
   WHERE r.moderated_at IS NULL
";
if ($hasVerified) { $sql .= " AND r.verified_at IS NOT NULL "; }
$sql .= " ORDER BY COALESCE(r.verified_at, r.id) DESC LIMIT 30";

$rows = [];
try {
  $rows = db()->query($sql)->fetchAll();
} catch (Throwable $e) {
  echo "<div class='alert alert-error'>Error loading moderation queue: ".e($e->getMessage())."</div>";
  include __DIR__ . '/templates/footer.php'; exit;
}
?>
<h1>Moderation queue</h1>
<p class="small muted">These are graded results that were verified and are waiting for a moderator to lock them.</p>

<?php if (!$rows): ?>
  <div class="alert alert-ok">Nothing to moderate right now 🎉</div>
<?php else: ?>
  <form method="post" action="<?= e(url('moderate_action.php')) ?>">
    <table class="mini">
      <tr>
        <th>Learner</th>
        <th>Course</th>
        <th>Decision</th>
        <?php if ($hasVerified): ?><th>Verified at</th><?php endif; ?>
        <th style="width:40%;">Notes</th>
        <th>Action</th>
      </tr>
      <?php foreach($rows as $r): ?>
        <tr>
          <td>
            <?= e($r['learner_name']) ?><br>
            <span class="small muted"><?= e($r['learner_email']) ?></span>
          </td>
          <td><?= e($r['course_title']) ?></td>
          <td><strong><?= e($r['decision']) ?></strong></td>
          <?php if ($hasVerified): ?>
            <td><?= e($r['verified_at'] ?: '—') ?></td>
          <?php endif; ?>
          <td>
            <textarea name="notes[<?= (int)$r['id'] ?>]" rows="2" style="width:100%;" placeholder="Sampling notes / reasons / checklist points..."></textarea>
          </td>
          <td>
            <input type="hidden" name="result_ids[]" value="<?= (int)$r['id'] ?>">
            <button class="btn btn-small" name="action" value="lock-one-<?= (int)$r['id'] ?>">Lock</button>
          </td>
        </tr>
      <?php endforeach; ?>
    </table>

    <div class="form__actions" style="margin-top:10px;">
      <button class="btn btn-primary" name="action" value="lock-all">Lock all listed</button>
      <a class="btn btn-outline" href="<?= e(url('dashboard.php')) ?>">Back</a>
    </div>
  </form>
<?php endif; ?>

<?php include __DIR__ . '/templates/footer.php'; ?>
