<?php
// profile.php — learner profile (name + SA ID + certified ID doc upload)
ini_set('display_errors',1); error_reporting(E_ALL);

require_once __DIR__.'/lib/auth.php';
require_once __DIR__.'/lib/db.php';
require_once __DIR__.'/lib/helpers.php';
require_once __DIR__.'/lib/csrf.php';
require_once __DIR__.'/lib/pii.php';

require_login();
$u   = current_user();
$msg = ''; $err = '';

/* ---------- helpers ---------- */
function za_id_is_valid(string $id): bool {
  // Must be exactly 13 digits; apply Luhn (common ZA variant)
  if (!preg_match('/^\d{13}$/', $id)) return false;
  $sum = 0; $alt = false;
  for ($i = 12; $i >= 0; $i--) {
    $n = intval($id[$i], 10);
    if ($alt) { $n *= 2; if ($n > 9) $n -= 9; }
    $sum += $n; $alt = !$alt;
  }
  return ($sum % 10) === 0;
}
//function pii_last4(string $id): string { return substr(preg_replace('/\D+/','',$id), -4) ?: ''; }
function storage_ids_dir_abs(): string { return __DIR__.'/storage/ids'; } // /lmis/storage/ids
function ensure_dir(string $path): void { if (!is_dir($path)) @mkdir($path, 0775, true); }

/* ---------- read current (for display) ---------- */
$st = db()->prepare("SELECT * FROM users WHERE id=? LIMIT 1");
$st->execute([$u['id']]); $u = $st->fetch();

/* Masked ID for placeholder */
$current_id_mask = '';
try {
  if (!empty($u['national_id_enc'])) {
    $plain = pii_decrypt($u['national_id_enc']);
    if ($plain && preg_match('/^\d{13}$/', $plain)) {
      $current_id_mask = substr($plain, 0, 6) . '******' . substr($plain, -1);
    }
  }
} catch (Throwable $e) { /* ignore */ }

/* ---------- handle POST ---------- */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  verify_csrf_or_die();

  $name  = trim($_POST['name'] ?? ($u['name'] ?? ''));
  $id_in = trim($_POST['national_id'] ?? ''); // may be blank if user keeps old

  if ($name === '') {
    $err = "Name is required.";
  }

  // Validate ID if user typed anything
  $enc = null; $last4 = null;
  if (!$err && $id_in !== '') {
    $digits = preg_replace('/\D+/', '', $id_in);
    if (!za_id_is_valid($digits)) {
      $err = "Please enter a valid 13-digit South African ID number (digits only).";
    } else {
      $enc  = pii_encrypt($digits);
      $last4 = pii_last4($digits);
    }
  }

  // Optional: certified ID document upload
  $docRelPath = null; // if new upload succeeds, store relative path
  if (!$err && isset($_FILES['national_id_doc']) && is_uploaded_file($_FILES['national_id_doc']['tmp_name'])) {
    $f = $_FILES['national_id_doc'];

    // Size cap ~5MB
    if ($f['size'] > 5*1024*1024) {
      $err = "File too large. Max 5 MB.";
    } else {
      // Basic MIME allow-list
      $allowed = ['application/pdf','image/jpeg','image/png'];
      $fi = new finfo(FILEINFO_MIME_TYPE);
      $mime = $fi->file($f['tmp_name']) ?: '';
      if (!in_array($mime, $allowed, true)) {
        $err = "Unsupported file type. Please upload a PDF, JPG or PNG.";
      } else {
        // Build safe destination: /lmis/storage/ids/{user_id}/id_doc.ext
        $baseDir = storage_ids_dir_abs() . '/' . (int)$u['id'];
        ensure_dir($baseDir);

        $ext = ($mime==='application/pdf') ? 'pdf' : (($mime==='image/jpeg') ? 'jpg' : 'png');
        $destAbs = $baseDir . '/id_doc.' . $ext;

        if (!@move_uploaded_file($f['tmp_name'], $destAbs)) {
          $err = "Server could not save your document. Please try again.";
        } else {
          // Web path (relative to /lmis)
          $docRelPath = 'storage/ids/' . (int)$u['id'] . '/id_doc.' . $ext;
        }
      }
    }
  }

  if (!$err) {
    // Update name
    $s = db()->prepare("UPDATE users SET name=? WHERE id=?");
    $s->execute([$name, $u['id']]);

    // Update ID if provided
    if ($enc !== null) {
      $s = db()->prepare("UPDATE users SET national_id_enc=?, national_id_last4=? WHERE id=?");
      $s->execute([$enc, $last4, $u['id']]);
    }

    // Update document path if uploaded
    if ($docRelPath !== null) {
      $s = db()->prepare("UPDATE users SET national_id_doc_path=? WHERE id=?");
      $s->execute([$docRelPath, $u['id']]);
    }

    $msg = "Saved.";

    // refresh user row for display
    $st = db()->prepare("SELECT * FROM users WHERE id=? LIMIT 1");
    $st->execute([$u['id']]); $u = $st->fetch();

    // refresh masked
    $current_id_mask = '';
    try {
      if (!empty($u['national_id_enc'])) {
        $plain = pii_decrypt($u['national_id_enc']);
        if ($plain && preg_match('/^\d{13}$/', $plain)) {
          $current_id_mask = substr($plain, 0, 6) . '******' . substr($plain, -1);
        }
      }
    } catch (Throwable $e) { /* ignore */ }
  }
}

/* ---------- render ---------- */
include __DIR__ . '/templates/header.php';
?>
<h1>My Profile</h1>

<?php if ($msg): ?><div class="alert alert-ok"><?= e($msg) ?></div><?php endif; ?>
<?php if ($err): ?><div class="alert alert-error"><?= e($err) ?></div><?php endif; ?>

<form method="post" enctype="multipart/form-data" class="form-card" style="max-width:560px;">
  <?php csrf_field(); ?>

  <label class="field">
    <span class="field__label">Full Name</span>
    <input class="field__input" name="name" value="<?= e($u['name'] ?? '') ?>" required>
  </label>

  <label class="field">
    <span class="field__label">South African ID (13 digits)</span>
    <input class="field__input"
           name="national_id"
           inputmode="numeric"
           pattern="\d{13}"
           maxlength="13"
           placeholder="<?= $current_id_mask ? e($current_id_mask) : 'e.g. 8001015009087' ?>">
    <div class="small muted">We encrypt this at rest. Required for certificate printing.</div>
  </label>

  <label class="field">
    <span class="field__label">Certified ID Copy (PDF/JPG/PNG)</span>
    <input class="field__input" type="file" name="national_id_doc" accept=".pdf,.jpg,.jpeg,.png">
    <div class="small muted">Max 5 MB. Clear scan or photo of the certified copy.</div>
    <?php if (!empty($u['national_id_doc_path'])): ?>
      <div class="small" style="margin-top:6px;">
        Current file: <a href="<?= e(url($u['national_id_doc_path'])) ?>" target="_blank">View / Download</a>
      </div>
    <?php endif; ?>
  </label>

  <div class="form__actions">
    <button class="btn btn-primary" type="submit">Save</button>
    <a class="btn btn-outline" href="<?= e(url('dashboard.php')) ?>">Back</a>
  </div>
</form>

<?php include __DIR__ . '/templates/footer.php'; ?>
