<?php
// Driver Analytics — latest scores + training recommendations
// Path: public_html/lmis/reports/driver_analytics.php

require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/db.php';
require_once __DIR__ . '/../lib/helpers.php';

require_login();            // must be logged in
require_ability('report');  // or 'admin' — adjust if needed

// --- Optional filters (KISS) ---
$q = trim($_GET['q'] ?? '');           // search by name/email/category
$cat = trim(strtolower($_GET['cat'] ?? ''));  // filter by category (e.g. speeding)

// Build SQL with params (latest score per user/category)
$sql = "
SELECT u.id AS user_id, u.name, u.email,
       ds.category, ds.score, ds.created_at,
       co.id AS rec_course_id, co.title AS rec_course
FROM driver_scores ds
JOIN users u ON u.id = ds.user_id
LEFT JOIN score_training_map stm
  ON stm.category = ds.category
  AND ds.score BETWEEN stm.min_score AND stm.max_score
LEFT JOIN courses co ON co.id = stm.course_id
WHERE ds.created_at = (
  SELECT MAX(created_at) FROM driver_scores
  WHERE user_id = ds.user_id AND category = ds.category
)
";
$params = [];

// apply filters safely
if ($q !== '') {
  $sql .= " AND (u.name LIKE ? OR u.email LIKE ? OR ds.category LIKE ?) ";
  $like = '%'.$q.'%';
  array_push($params, $like, $like, $like);
}
if ($cat !== '') {
  $sql .= " AND ds.category = ? ";
  $params[] = $cat;
}

$sql .= " ORDER BY u.name, ds.category ";

$st = db()->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll();

// for category dropdown (distinct categories)
$cats = db()->query("SELECT DISTINCT category FROM driver_scores ORDER BY category")->fetchAll();

include __DIR__ . '/../templates/header.php';
?>
<h1>Driver Analytics — Latest Scores & Training Recommendations</h1>
<p class="small">
  This page shows each learner/driver’s latest score for each risk category (like <em>speeding</em> or <em>fatigue</em>),
  and the course we recommend based on your mapping rules.
</p>

<form method="get" class="form-card" style="margin-bottom:12px;">
  <div class="grid-2">
    <label class="field">
      <span class="field__label">Search (name, email, category)</span>
      <input class="field__input" name="q" value="<?= e($q) ?>" placeholder="e.g. thabo, fatigue">
    </label>
    <label class="field">
      <span class="field__label">Category</span>
      <select class="field__input" name="cat">
        <option value="">All categories</option>
        <?php foreach($cats as $c): $v=$c['category']; ?>
          <option value="<?= e($v) ?>" <?= $cat===$v ? 'selected' : '' ?>><?= e($v) ?></option>
        <?php endforeach; ?>
      </select>
    </label>
  </div>
  <div class="form__actions">
    <button class="btn btn-primary btn-small">Filter</button>
    <a class="btn btn-small btn-outline" href="<?= e(url('reports/driver_analytics.php')) ?>">Reset</a>
    <?php if (can('admin')): ?>
      <a class="btn btn-small" href="<?= e(url('admin/driver_scores_upload.php')) ?>">Upload Scores (CSV)</a>
      <a class="btn btn-small" href="<?= e(url('admin/driver_scores_mapping.php')) ?>">Score → Training Rules</a>
    <?php endif; ?>
  </div>
</form>

<div class="panel">
  <div class="panel__body">
    <table class="mini">
      <tr>
        <th>Driver</th>
        <th>Email</th>
        <th>Category</th>
        <th>Score</th>
        <th>When</th>
        <th>Recommended Course</th>
        <th>Action</th>
      </tr>
      <?php if (!$rows): ?>
        <tr><td colspan="7" class="muted">No scores found yet. Upload a CSV in <em>Upload Scores</em>, or enter scores manually in your admin tools.</td></tr>
      <?php endif; ?>

      <?php foreach ($rows as $r): ?>
      <tr>
        <td><?= e($r['name']) ?></td>
        <td><?= e($r['email']) ?></td>
        <td><?= e($r['category']) ?></td>
        <td><?= (int)$r['score'] ?></td>
        <td><?= e($r['created_at']) ?></td>
        <td><?= $r['rec_course'] ? e($r['rec_course']) : '<span class="muted">No mapping for this score</span>' ?></td>
        <td>
          <?php if ($r['rec_course_id']): ?>
            <form method="post" action="<?= e(url('admin/enroll_recommendation.php')) ?>" style="display:inline">
              <input type="hidden" name="user_id" value="<?= (int)$r['user_id'] ?>">
              <input type="hidden" name="course_id" value="<?= (int)$r['rec_course_id'] ?>">
              <button class="btn btn-small">Enroll</button>
            </form>
          <?php else: ?>
            <?php if (can('admin')): ?>
              <a class="btn btn-small btn-outline" href="<?= e(url('admin/driver_scores_mapping.php')) ?>">Add mapping</a>
            <?php else: ?>
              <span class="muted">—</span>
            <?php endif; ?>
          <?php endif; ?>
        </td>
      </tr>
      <?php endforeach; ?>
    </table>
  </div>
</div>

<?php include __DIR__ . '/../templates/footer.php'; ?>
